<?php
/**
 * Copyright (с) Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2025 All Rights Reserved
 *
 * Licensed under CLOUD LINUX LICENSE AGREEMENT
 * https://www.cloudlinux.com/legal/
 */

namespace CloudLinux\Imunify\App\Model;

/**
 * Malware item model.
 */
class MalwareItem {

	/**
	 * Detection timestamp.
	 *
	 * @var int
	 */
	private $detectedAt;

	/**
	 * File.
	 *
	 * @var string
	 */
	private $file;

	/**
	 * Resource type.
	 *
	 * @var string
	 */
	private $resourceType;

	/**
	 * Malicious.
	 *
	 * @var bool
	 */
	private $malicious;

	/**
	 * Status.
	 *
	 * @var string
	 */
	private $status;

	/**
	 * Cleaned at timestamp.
	 *
	 * @var int|null
	 */
	private $cleanedAt;

	/**
	 * Malware signature.
	 *
	 * @var string
	 */
	private $signature;

	/**
	 * Database name.
	 *
	 * @var string
	 */
	private $dbName;

	/**
	 * Check if the resource type is file.
	 *
	 * @return bool
	 */
	private function isFile() {
		return 'file' === $this->resourceType;
	}
	/**
	 * Get the path.
	 *
	 * File path if the resource type is file, database name otherwise.
	 *
	 * @return string
	 */
	public function getPath() {
		return $this->isFile() ? $this->file : $this->dbName;
	}

	/**
	 * Get the signature.
	 *
	 * @return string
	 */
	public function getSignature() {
		return $this->signature;
	}

	/**
	 * Get the detection timestamp.
	 *
	 * @return int
	 */
	public function getDetectedAt() {
		return $this->detectedAt;
	}

	/**
	 * Get the file.
	 *
	 * @return string
	 */
	public function getFile() {
		return $this->file;
	}

	/**
	 * Get the resource type.
	 *
	 * @return string
	 */
	public function getResourceType() {
		return $this->resourceType;
	}

	/**
	 * Get if the item is malicious.
	 *
	 * @return bool
	 */
	public function isMalicious() {
		return $this->malicious;
	}

	/**
	 * Get the status.
	 *
	 * @return string
	 */
	public function getStatus() {
		return $this->status;
	}

	/**
	 * Get the status label.
	 *
	 * @return string
	 *
	 * @since 2.0.0
	 */
	public function getStatusLabel() {
		switch ( $this->status ) {
			case 'cleanup_done':
				return esc_html__( 'Cleaned', 'imunify-security' );
			case 'cleanup_removed':
				return esc_html__( 'Content removed', 'imunify-security' );
			case 'cleanup_started':
			case 'cleanup_pending':
				return esc_html__( 'Cleanup in progress', 'imunify-security' );
			case 'found':
			case 'infected':
			case 'cleanup_requires_myimunify_protection':
				return esc_html__( 'Infected', 'imunify-security' );
			case 'quarantined':
				return esc_html__( 'Quarantined', 'imunify-security' );
			default:
				return esc_html__( 'Unknown', 'imunify-security' );
		}
	}

	/**
	 * Get the status extra CSS class.
	 *
	 * @param string $baseClass Base CSS class.
	 *
	 * @return string
	 *
	 * @since 2.0.0
	 */
	public function getStatusExtraCssClass( $baseClass ) {
		switch ( $this->status ) {
			case 'found':
			case 'infected':
			case 'cleanup_requires_myimunify_protection':
				return $baseClass . '--infected';
			default:
				return '';
		}
	}

	/**
	 * Get the cleaned at timestamp.
	 *
	 * @return int|null
	 */
	public function getCleanedAt() {
		return $this->cleanedAt;
	}

	/**
	 * Get the last action timestamp (cleaned or detected).
	 *
	 * @return int
	 *
	 * @since 2.0.0
	 */
	public function getLastActionDate() {
		return is_null( $this->cleanedAt ) ? $this->detectedAt : $this->cleanedAt;
	}

	/**
	 * Check if the malware was cleaned.
	 *
	 * @return bool True if the malware was cleaned.
	 */
	public function isCleaned() {
		return in_array( $this->status, array( 'cleanup_removed', 'cleanup_done' ), true );
	}

	/**
	 * Convert to array
	 *
	 * @return array
	 */
	public function toArray() {
		return array(
			'created'       => $this->detectedAt,
			'file'          => $this->file,
			'resource_type' => $this->resourceType,
			'malicious'     => $this->malicious,
			'status'        => $this->status,
			'cleaned_at'    => $this->cleanedAt,
			'type'          => $this->signature,
			'db_name'       => $this->dbName,
		);
	}

	/**
	 * Create from array
	 *
	 * @param array $data Data.
	 *
	 * @return self
	 */
	public static function fromArray( $data ) {
		$result               = new self();
		$result->detectedAt   = isset( $data['created'] ) ? intval( $data['created'] ) : 0;
		$result->file         = isset( $data['file'] ) ? $data['file'] : '';
		$result->resourceType = isset( $data['resource_type'] ) ? $data['resource_type'] : '';
		$result->malicious    = isset( $data['malicious'] ) && (bool) $data['malicious'];
		$result->status       = isset( $data['status'] ) ? $data['status'] : '';
		$result->cleanedAt    = isset( $data['cleaned_at'] ) ? intval( $data['cleaned_at'] ) : null;
		$result->signature    = isset( $data['type'] ) ? $data['type'] : '';
		$result->dbName       = isset( $data['db_name'] ) ? $data['db_name'] : '';

		return $result;
	}
}
