#!/usr/bin/bash

set -eu -o pipefail

KERNELCARE_CERT_WRAPPER="/usr/share/kcare/secure_boot/shim_certificate.efi"

if [ "$EUID" -ne 0 ]; then
    >&2 echo "$0 should be run as root"
    exit 1
fi

if [ ! -d /boot/efi/EFI ]; then
    >&2 echo "No EFI directory found. Please ensure /boot/efi is mounted."
    exit 1
fi

readarray -t shim_binary < <(find /boot/efi/EFI -name "shim[^_]*.efi" -type f)
if [ ${#shim_binary[@]} -eq 0 ]; then
    >&2 echo "No shim binary found. Please ensure shim is installed."
    exit 1
fi

shim_dir=$(dirname "${shim_binary[0]}")
for shim in "${shim_binary[@]}"; do
    if [ "$(dirname "$shim")" != "$shim_dir" ]; then
        >&2 echo "Confused: multiple shim locations found:"
        >&2 print "  %s\n" "${shim_binary[@]}"
        exit 1
    fi
done

cert_wrapper="${shim_dir}/shim_certificate.efi"
if [ -f "$cert_wrapper" ]; then
    if grep -aq 'Cloud Linux Software' "$cert_wrapper"; then
        echo "Found KernelCare certificates at $cert_wrapper, updating..."
        cp "$KERNELCARE_CERT_WRAPPER" "$cert_wrapper"
    else
        >&2 echo "File $cert_wrapper already exists, cannot rewrite it."
        exit 1
    fi
else
    echo "Installing KernelCare certificates to $cert_wrapper"
    cp "$KERNELCARE_CERT_WRAPPER" "$cert_wrapper"
fi

echo "Done! Please reboot for the changes to take effect."
